import numpy as np
import torch.nn.functional as F
import torch

def get_2d_sincos_pos_embed_for_coords(embed_dim, coords, image_size=224, patch_size=16):
    """
    embed_dim: int, the embedding dimension
    coords: an array of shape [num_pixels, 2], containing the x, y coordinates of the pixels (on the raw image)
    image_size: size of the rgb image 
    patch size: size of the patch
    return:
    pos_embed: [num_pixels, embed_dim]
    """
    assert embed_dim % 2 == 0

    grid_x = coords[:, 0] - patch_size // 2
    grid_y = coords[:, 1] - patch_size // 2
    
    emb_x = get_1d_sincos_pos_embed_from_grid(embed_dim // 2, grid_x, idx_range=image_size - patch_size)  
    emb_y = get_1d_sincos_pos_embed_from_grid(embed_dim // 2, grid_y, idx_range=image_size - patch_size)  

    emb = np.concatenate([emb_x, emb_y], axis=1)  
    return emb

def get_2d_sincos_pos_embed(embed_dim, grid_size, cls_token=False):
    """
    grid_size: int of the grid height and width
    return:
    pos_embed: [grid_size*grid_size, embed_dim] or [1+grid_size*grid_size, embed_dim] (w/ or w/o cls_token)
    """
    grid_h = np.arange(grid_size, dtype=np.float32)
    grid_w = np.arange(grid_size, dtype=np.float32)
    grid = np.meshgrid(grid_w, grid_h)  
    grid = np.stack(grid, axis=0)

    grid = grid.reshape([2, 1, grid_size, grid_size])
    pos_embed = get_2d_sincos_pos_embed_from_grid(embed_dim, grid)
    if cls_token:
        pos_embed = np.concatenate([np.zeros([1, embed_dim]), pos_embed], axis=0)
    return pos_embed


def get_2d_sincos_pos_embed_from_grid(embed_dim, grid):
    assert embed_dim % 2 == 0

    emb_h = get_1d_sincos_pos_embed_from_grid(embed_dim // 2, grid[0])  
    emb_w = get_1d_sincos_pos_embed_from_grid(embed_dim // 2, grid[1])  

    emb = np.concatenate([emb_h, emb_w], axis=1) 
    return emb


def get_1d_sincos_pos_embed_from_grid(embed_dim, pos, idx_range=14):
    """
    embed_dim: output dimension for each position
    pos: a list of positions to be encoded: size (M,)
    out: (M, D)
    """
    assert embed_dim % 2 == 0
    omega = np.arange(embed_dim // 2, dtype=np.float32)
    omega /= embed_dim / 2.
    
    omega = (1. / 10000**omega) / (idx_range - 1) * 13  

    pos = pos.reshape(-1)  
    out = np.einsum('m,d->md', pos, omega)  

    emb_sin = np.sin(out) 
    emb_cos = np.cos(out) 

    emb = np.concatenate([emb_sin, emb_cos], axis=1)  
    return emb


def interpolate_pos_embed(model, checkpoint_model):
    if 'pos_embed' in checkpoint_model:
        pos_embed_checkpoint = checkpoint_model['pos_embed']
        embedding_size = pos_embed_checkpoint.shape[-1]
        num_patches = model.patch_embed.num_patches
        num_extra_tokens = model.pos_embed.shape[-2] - num_patches
        
        orig_size = int((pos_embed_checkpoint.shape[-2] - num_extra_tokens) ** 0.5)
        
        new_size = int(num_patches ** 0.5)
        
        if orig_size != new_size:
            print("Position interpolate from %dx%d to %dx%d" % (orig_size, orig_size, new_size, new_size))
            extra_tokens = pos_embed_checkpoint[:, :num_extra_tokens]
            
            pos_tokens = pos_embed_checkpoint[:, num_extra_tokens:]
            pos_tokens = pos_tokens.reshape(-1, orig_size, orig_size, embedding_size).permute(0, 3, 1, 2)
            pos_tokens = torch.nn.functional.interpolate(
                pos_tokens, size=(new_size, new_size), mode='bicubic', align_corners=False)
            pos_tokens = pos_tokens.permute(0, 2, 3, 1).flatten(1, 2)
            new_pos_embed = torch.cat((extra_tokens, pos_tokens), dim=1)
            checkpoint_model['pos_embed'] = new_pos_embed


def interpolate_time_embed(model, checkpoint_model):

    if 't_embedder' in checkpoint_model and hasattr(model, 't_embedder'):
        checkpoint_time_embed = checkpoint_model['t_embedder']
        model_time_embed = model.t_embedder.mlp[0].weight  

        if checkpoint_time_embed.dim() == 2 and model_time_embed.dim() == 2:
            new_shape = model_time_embed.shape
            interpolated_time_embed = F.interpolate(checkpoint_time_embed.unsqueeze(0), size=new_shape[1], mode='linear', align_corners=True).squeeze(0)
            checkpoint_model['t_embedder'] = interpolated_time_embed
        else:
            raise ValueError("Invalid dimensions for time embedding.")